import Fastify from "fastify";
import cors from "@fastify/cors";
// import 'dotenv/config';
import dotenv from "dotenv";
import { fileURLToPath } from "url";
import path from "path";
const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);
dotenv.config({ path: path.join(__dirname, "../.env") });
import { ZodError, z } from "zod";
import { QdrantClient } from "@qdrant/js-client-rest";
import OpenAI from "openai";
import { MongoClient as MongoDbClient } from "mongodb";
const app = Fastify({ logger: true });
await app.register(cors, { origin: true });
const client = new OpenAI({ apiKey: process.env.OPENAI_API_KEY });
const qdrant = new QdrantClient({ url: process.env.QDRANT_URL || "http://localhost:6333" });
const COLLECTION = "epics_v1"; // create & ingest separately per your pipeline
const ReqSchema = z.object({
    query: z.string().min(1),
    lang: z.enum(["en", "hi", "auto"]).default("auto"),
    k: z.number().min(1).max(4).default(4)
});
function trim(s, n = 500) { return s.length <= n ? s : s.slice(0, n - 1) + "…"; }
async function embed(query) {
    const r = await client.embeddings.create({ model: "text-embedding-3-large", input: query });
    return r.data[0].embedding;
}
let _mongoClient = null;
let _mongoDb = null;
async function getMongoDb() {
    if (_mongoDb)
        return _mongoDb;
    const uri = process.env.MONGO_URI;
    const dbName = process.env.MONGO_DB || "voice_therapist";
    if (!uri)
        throw new Error("Missing MONGO_URI in apps/kb/.env");
    _mongoClient = new MongoDbClient(uri);
    await _mongoClient.connect();
    _mongoDb = _mongoClient.db(dbName);
    app.log.info({ dbName }, "KB Mongo connected");
    return _mongoDb;
}
app.post("/kb/search", async (req, res) => {
    const db = await getMongoDb().catch(() => null);
    if (req.headers["x-crisis"] === "1") {
        if (db) {
            await db.collection("kb_queries").insertOne({
                ts: new Date(),
                crisis: true,
                query: req.body?.query ?? null,
                lang: req.body?.lang ?? null,
                k: req.body?.k ?? null,
                returned: 0,
                source: "bypass_crisis",
            });
        }
        return res.send({ passages: [] });
    }
    let parsed;
    try {
        parsed = ReqSchema.parse(req.body || {});
    }
    catch (e) {
        if (e instanceof ZodError)
            return res.code(400).send({ error: e.errors });
        throw e;
    }
    const vector = await embed(parsed.query);
    const start = Date.now();
    let results = [];
    try {
        const rr = await qdrant.search(COLLECTION, {
            vector, limit: 30, with_payload: true, score_threshold: 0.05,
            // language filter would be: filter: { must: [{ key:"language", match:{ value: parsed.lang }}] }
        });
        results = rr;
    }
    catch (e) {
        // If collection not found or Qdrant not running, return a tiny curated fallback.
        return res.send({ passages: [
                { passage: "Sometimes we feel torn between duty and compassion. Many find it helpful to pause, breathe, and choose the path that reduces harm—to self and others.",
                    source: { work: "general", book: "", chapter: "", verse_range: "", edition: "" },
                    language: parsed.lang, sensitive_tags: [] }
            ] });
    }
    const top = results.slice(0, parsed.k).map((r) => ({
        passage: trim(r.payload.text || ""),
        source: {
            work: r.payload.work, book: r.payload.book, chapter: r.payload.chapter,
            verse_range: r.payload.verse_range, edition: r.payload.source_edition
        },
        language: r.payload.language,
        sensitive_tags: r.payload.sensitive_tags || []
    }));
    const latency = Date.now() - start;
    res.header("x-kb-latency-ms", String(latency));
    if (db) {
        await db.collection("kb_queries").insertOne({
            ts: new Date(),
            crisis: false,
            query: parsed.query,
            lang: parsed.lang,
            k: parsed.k,
            returned: top.length,
            latencyMs: latency,
            collection: COLLECTION,
            passages: top.map(p => ({
                passage: p.passage,
                source: p.source,
                language: p.language,
                sensitive_tags: p.sensitive_tags,
            })),
        });
    }
    return res.send({ passages: top });
});
app.get("/healthz", async () => ({ ok: true }));
process.on("SIGINT", async () => {
    if (_mongoClient)
        await _mongoClient.close().catch(() => { });
    process.exit(0);
});
process.on("SIGTERM", async () => {
    if (_mongoClient)
        await _mongoClient.close().catch(() => { });
    process.exit(0);
});
app.listen({ port: 8081, host: "0.0.0.0" }).then(() => {
    console.log("KB service on :8081");
}).catch((err) => {
    console.error(err);
    process.exit(1);
});
