import { FastifyInstance, FastifyPluginAsync } from "fastify";
import { logger, SessionState } from "@starter/shared";
import { VapiAdapter } from "../telephony/vapi";
import { getDb } from "../db";
import { CFG } from "../config";
import { redactPII } from "../redact";
import { encryptMaybe } from "../crypto";
import { CallService } from "../services/callService.js";
import {createHash} from "crypto";
import type { InjectOptions } from "fastify";


// Sessions (ephemeral, per live call)
const sessions = new Map<string, SessionState>();
const vapi = new VapiAdapter({ apiKey: CFG.vapiKey });
const callService = CallService.getInstance();
let lastEventAt: number | null = null;

type MyInjectOptions = Omit<InjectOptions, "method"> & {
  method: "GET" | "POST" | "PUT" | "PATCH" | "DELETE" | "OPTIONS" | "HEAD";
};


const KB_COOLDOWN_MS = 2 * 60_000;
const MAX_KB_PER_CALL = 2;

async function ensureCall(callId: string) {
  if (!CFG.storeMetadata) return;
  const db = await getDb();
  await db.collection("calls").updateOne(
    { callId },
    {
      $setOnInsert: {
        callId,
        startedAt: new Date(),
        lang: "en",
        transcriptOptIn: true, // set true only after consent
      },
    },
    { upsert: true }
  );
}

async function saveUserUtterance(callId: string, transcript: string, isFinal = true) {
  if (!CFG.storeTranscripts || !transcript) return;
  if (!isFinal) return; // only store finals 

  const db = await getDb();
  const call = await db.collection("calls").findOne(
    { callId },
    { projection: { transcriptOptIn: 1 } }
  );
  if (!call?.transcriptOptIn) return; // only if user consented

  const clean = redactPII(transcript);
  const { ciphertext, iv, tag } = encryptMaybe(clean);
  const expiresAt = new Date(Date.now() + CFG.ttlDays * 24 * 60 * 60 * 1000);

  const hash = textHash(clean);

  await db.collection("utterances").updateOne(
  { callId, role: "user", hash, isFinal: true },
  {
    $setOnInsert: {
      callId,
      role: "user",
      hash,
      ts: new Date(),
      expiresAt,
      text: CFG.enc.enabled ? undefined : clean,
      enc: CFG.enc.enabled ? { ciphertext, iv, tag } : null,
      isFinal: true,
      meta: { source: "vapi", lang: "en" },
    },
  },
  { upsert: true }
);
}

// store assistant side too
export async function saveAssistantUtterance(callId: string, text: string, isFinal = true) {
  if (!CFG.storeTranscripts || !text) return;
  if (!isFinal) return; //only store finals

  const db = await getDb();
  const call = await db.collection("calls").findOne(
    { callId },
    { projection: { transcriptOptIn: 1 } }
  );
  if (!call?.transcriptOptIn) return;

  const clean = redactPII(text);
  const { ciphertext, iv, tag } = encryptMaybe(clean);
  const expiresAt = new Date(Date.now() + CFG.ttlDays * 24 * 60 * 60 * 1000);

  const hash = textHash(clean);


  await db.collection("utterances").updateOne(
  { callId, role: "assistant", hash, isFinal: true },
  {
    $setOnInsert: {
      callId,
      role: "assistant",
      hash,
      ts: new Date(),
      expiresAt,
      text: CFG.enc.enabled ? undefined : clean,
      enc: CFG.enc.enabled ? { ciphertext, iv, tag } : null,
      isFinal: true,
      meta: { source: "assistant", lang: "en" },
    },
  },
  { upsert: true }
);
}

function allowKb(s: SessionState) {
  return (
    s.kb_opt_in &&
    !s.crisis &&
    Date.now() - s.lastKbMs > KB_COOLDOWN_MS &&
    s.kbUses < MAX_KB_PER_CALL
  );
}

// for hashing: start
function normalizeForHash(s: string) {
  return String(s || "")
    .trim()
    .replace(/\s+/g, " ")
    .toLowerCase();
}

function textHash(s: string) {
  return createHash("sha256").update(normalizeForHash(s)).digest("hex");
}
// for hashing: end


/**
 * Build a single chronological transcript from utterances (user + assistant, finals only)
 * and store it on the calls record.
 *
 * - Stores as `chronologicalTranscript` (array) + `chronologicalTranscriptText` (string)
 * - If encryption enabled, stores encrypted versions `chronologicalTranscriptEnc` and `chronologicalTranscriptTextEnc`
 */
async function buildAndStoreChronologicalTranscript(callId: string) {
  try {
    const db = await getDb();

    const call = await db.collection("calls").findOne(
      { callId },
      { projection: { transcriptOptIn: 1 } }
    );
    if (!call?.transcriptOptIn) return; // respect consent

    // Pull only finals; rely on ts ordering for flow
    const turns = await db
      .collection("utterances")
      .find({ callId, isFinal: { $ne: false } })
      .project({ role: 1, text: 1, enc: 1, ts: 1 })
      .sort({ ts: 1 })
      .toArray();

    // If enc enabled, we can't decrypt here (no decrypt in this process). Store encrypted JSON blobs.
    if (CFG.enc.enabled) {
      const safeArrayJson = JSON.stringify(
        turns.map(t => ({
          role: t.role,
          // keep text if present (it will be undefined when enc enabled at utterance level)
          text: t.text ?? null,
          // NOTE: enc is already encrypted per-utterance; we're keeping the pointer only
          enc: t.enc ?? null,
          ts: t.ts,
        }))
      );
      const linesText = turns
        .map(t => {
          const label = t.role === "assistant" ? "Assistant" : "User";
          return `${label}: ${t.text ?? "[encrypted]"}`;
        })
        .join("\n");

      const transArrayEnc = encryptMaybe(safeArrayJson);
      const transTextEnc = encryptMaybe(linesText);

      await db.collection("calls").updateOne(
        { callId },
        {
          $set: {
            transcriptReadyAt: new Date(),
            chronologicalTranscriptEnc: {
              ciphertext: transArrayEnc.ciphertext,
              iv: transArrayEnc.iv,
              tag: transArrayEnc.tag,
            },
            chronologicalTranscriptTextEnc: {
              ciphertext: transTextEnc.ciphertext,
              iv: transTextEnc.iv,
              tag: transTextEnc.tag,
            },
          },
        },
        { upsert: true }
      );
      return;
    }

    // Non-encrypted path (plain text, already PII-redacted at save time)
    const chronologicalTranscript = turns.map(t => ({
      role: t.role,
      text: t.text || "",
      ts: t.ts,
    }));
    const chronologicalTranscriptText = chronologicalTranscript
      .map(t => `${t.role === "assistant" ? "Assistant" : "User"}: ${t.text}`)
      .join("\n");

    await db.collection("calls").updateOne(
      { callId },
      {
        $set: {
          transcriptReadyAt: new Date(),
          chronologicalTranscript,
          chronologicalTranscriptText,
        },
      },
      { upsert: true }
    );
  } catch (e) {
    logger.error({ e, callId }, "Failed to build/store chronological transcript");
  }
}

function parseStitchedTranscriptToTurns(stitched: any): Array<{ role: "user" | "assistant"; text: string }> {
  if (!stitched) return [];

  // If Vapi sends structured transcript (array of turns)
  // common shapes: [{ role: 'user', text: '...' }, { role:'assistant', text:'...' }]
  if (Array.isArray(stitched)) {
    return stitched
      .map((t: any) => ({
        role: (t.role === "assistant" ? "assistant" : "user") as "user" | "assistant",
        text: String(t.text ?? t.content ?? t.transcript ?? "").trim(),
      }))
      .filter(t => t.text);
  }

  // If it's an object containing a field
  if (typeof stitched === "object") {
    const maybe = stitched.text ?? stitched.transcript ?? stitched.content;
    if (typeof maybe === "string") return parseStitchedTranscriptToTurns(maybe);
  }

  // Plain string transcript: "User: ...\nAssistant: ...\nUser: ..."
  const s = String(stitched).trim();
  if (!s) return [];

  const lines = s.split(/\r?\n/).map(l => l.trim()).filter(Boolean);

  const turns: Array<{ role: "user" | "assistant"; text: string }> = [];
  let currentRole: "user" | "assistant" | null = null;
  let buf: string[] = [];

  const flush = () => {
    const text = buf.join(" ").trim();
    if (currentRole && text) turns.push({ role: currentRole, text });
    buf = [];
  };

  const roleFromLine = (line: string): "user" | "assistant" | null => {
    if (/^(user|caller|patient)\s*[:\-]/i.test(line)) return "user";
    if (/^(assistant|agent|therapist)\s*[:\-]/i.test(line)) return "assistant";
    return null;
  };

  for (const line of lines) {
    const r = roleFromLine(line);
    if (r) {
      flush();
      currentRole = r;
      buf.push(line.replace(/^(user|caller|patient|assistant|agent|therapist)\s*[:\-]\s*/i, "").trim());
    } else {
      if (!currentRole) currentRole = "user";
      buf.push(line);
    }
  }
  flush();

  // Merge consecutive turns with same role (prevents micro-fragment spam)
  const merged: typeof turns = [];
  for (const t of turns){
    const prev = merged[merged.length - 1];
    if (prev && prev.role === t.role) {
      prev.text = `${prev.text} ${t.text}`.trim();
    } else {
      merged.push(t);
    }
  }

  return merged;
}

async function upsertUtterancesFromStitchedTranscript(callId: string, stitchedTranscript: any) {
  if (!CFG.storeTranscripts) return;

  const db = await getDb();

  const call = await db.collection("calls").findOne(
    { callId },
    { projection: { transcriptOptIn: 1 } }
  );
  if (!call?.transcriptOptIn) return;

  const turns = parseStitchedTranscriptToTurns(stitchedTranscript);
  if (!turns.length) return;

  // Replace the "utterances" for this call with the stitched authoritative version.
  // This fixes your "only user utterances" problem permanently.
  await db.collection("utterances").deleteMany({ callId });

  const expiresAt = new Date(Date.now() + CFG.ttlDays * 24 * 60 * 60 * 1000);
  const base = Date.now();

  const docs = turns.map((t, i) => {
    const clean = redactPII(t.text);
    const { ciphertext, iv, tag } = encryptMaybe(clean);
    return {
      callId,
      role: t.role,
      ts: new Date(base + i), // stable chronological ordering
      expiresAt,
      text: CFG.enc.enabled ? undefined : clean,
      enc: CFG.enc.enabled ? { ciphertext, iv, tag } : null,
      isFinal: true,
      meta: { source: "vapi_end_of_call", lang: "en" },
    };
  });

  await db.collection("utterances").insertMany(docs, { ordered: true });
}


function sendInjected(res: any, r: any) {
  let payload: any = { ok: true };
  try {
    if (typeof r.body === "string" && r.body.length) payload = JSON.parse(r.body);
    else if (r.body) payload = r.body;
  } catch {
    payload = { ok: true };
  }
  return res.code(r.statusCode || 200).send(payload);
}


async function injectAndAck(app: FastifyInstance, req: any, res: any, opts: InjectOptions) {
  try {
    const r = await app.inject(opts);
    if (r.statusCode >= 400) {
      req.log.error(
        { status: r.statusCode, body: r.body, url: opts.url },
        "inject failed"
      );
    }
  } catch (e: any) {
    req.log.error({ err: e?.message, url: opts.url }, "inject threw");
  }

  // CRITICAL: never propagate internal errors to Vapi
  return res.code(200).send({ ok: true });
}


export const registerVapiRoutes: FastifyPluginAsync = async (app) => {
  const verify = (req: any, res: any) => {
    const secret = req.headers["x-vapi-secret"];
    if (process.env.VAPI_WEBHOOK_SECRET && secret !== process.env.VAPI_WEBHOOK_SECRET) {
      res.code(401).send({ ok: false, error: "unauthorized" });
      return false;
    }
    return true;
  };

  // -------------------------------
  // Call start
  // -------------------------------
  app.post("/vapi/call-start", async (req, res) => {
    const body: any = req.body || {};
    const callId = body.callId || body.call_id || `call_${Date.now()}`;
    
    // Extract phone numbers and other call details from Vapi webhook
    const fromNumber = body.customer?.number || body.from || body.caller?.number;
    const toNumber = body.phoneNumber?.number || body.to || body.assistant?.number;
    const vapiCallId = body.id || body.vapi_call_id;

    // Create comprehensive call record
    try {
      await callService.createCall({
        callId,
        vapiCallId,
        fromNumber,
        toNumber,
        startedAt: new Date(),
        lang: "auto", // will be updated as we detect language
        transcriptOptIn: true, // require explicit opt-in
      });
    } catch (error) {
      req.log.error({ error, callId }, "Failed to create call record");
      // Continue anyway - don't fail the call
    }

    sessions.set(callId, {
      callId,
      lang: "auto",
      kb_opt_in: false,
      crisis: false,
      lastKbMs: 0,
      kbUses: 0,
      turnCount: 0,
      metrics: {},
    });
    logger.info({ callId }, "Call started");
    return res.code(200).send({ ok: true });
  });

  // -------------------------------
  // User input (FINAL user turns routed here)
  // -------------------------------
  app.post("/vapi/user-input", async (req, res) => {
  const body: any = req.body || {};
  const callId = body.callId;
  const intent = body.intent;
  const transcript: string = body.transcript ?? "";
  const isFinal = !!body.isFinal;

  if (!callId) return res.code(400).send({ ok: false, error: "missing_callId" });

  // Session is optional; don't 404 if it's gone
  const s = sessions.get(callId);


  
  
  const safeUpdateCall = async (callId: string, patch: any) => {
    try {
      await callService.updateCall(callId, patch);
    } catch (e: any) {
      req.log.error({ err: e?.message, callId, patch }, "updateCall failed");
    }
  };
  // NEVER let DB issues break the call
  try {
    if (s) {
      if (intent === "opt_in_epics") {
        s.kb_opt_in = true;
        try {
          await safeUpdateCall(callId, { kb_opt_in: true });
        } catch (e) {
          req.log.error({ e, callId }, "updateCall failed: opt_in_epics");
        }
      }

      if (intent === "opt_in_transcripts") {
        try {
          await safeUpdateCall(callId, { transcriptOptIn: true });
        } catch (e) {
          req.log.error({ e, callId }, "updateCall failed: opt_in_transcripts");
        }
      }

      if (
        typeof transcript === "string" &&
        /suicide|kill myself|end my life|आत्महत्या|मरना/i.test(transcript)
      ) {
        s.crisis = true;
        try {
          await safeUpdateCall(callId, { crisis_flag: true, riskLevel: "high" });
        } catch (e) {
          req.log.error({ e, callId }, "updateCall failed: crisis_flag");
        }
      }

      // increment session counter (in-memory)
      s.turnCount += 1;

      // IMPORTANT: increment DB counter using $inc, NOT $set turnCount: s.turnCount
      // This prevents race issues and avoids the "$inc in replacement doc" failure path
      try {
        await callService.updateCall(callId, {}, { turnCount: 1 });
      } catch (e) {
        req.log.error({ e, callId }, "updateCall failed: turnCount $inc");
      }
    }
  } catch (e) {
    // catch-all safety: user input must never 500
    req.log.error({ e, callId }, "user-input session/db update block failed");
  }

  // Persist the user utterance (respects consent)
  try {
    if (transcript) await saveUserUtterance(callId, transcript, isFinal);
  } catch (e) {
    req.log.error({ e, callId }, "Failed to store user utterance");
  }

  // Always ACK
  return res.code(200).send({ ok: true });
});


  // -------------------------------
  // Tool calls (e.g., kb_search)
  // -------------------------------
  app.post("/vapi/tool-call", async (req, res) => {
    const body: any = req.body || {};
    const callId = body.callId;
    const tool = body.tool;

    if (!callId) return res.code(400).send({ ok: false, error: "missing_callId" });

    const s = sessions.get(callId);
    if (!s) {
      req.log.warn({ callId, tool }, "tool-call without session");
      return res.send({ ok: true, result: {} });
    }

    if (tool === "kb_search") {
      if (!allowKb(s)) {
        return res.send({ ok: true, result: { passages: [] } });
      }
      try {
        const r = await fetch(CFG.kbUrl, {
          method: "POST",
          headers: {
            "content-type": "application/json",
            "X-CRISIS": s.crisis ? "1" : "0",
          },
          body: JSON.stringify({ query: body.query, lang: s.lang, k: 4 }),
        });
        const data = await r.json();
        s.kbUses += 1;
        s.lastKbMs = Date.now();
        
        // Update database with KB usage
        await callService.updateCall(callId, { 
          kb_used: true, 
          kb_count: s.kbUses 
        });
        
        return res.send({ ok: true, result: data });
      } catch (e) {
        req.log.error({ e }, "kb_search failed");
        return res.send({ ok: true, result: { passages: [] } });
      }
    }

    if (tool === "crisis_signal") {
      const result = s.crisis
        ? { risk_level: "high", reason: "heuristic", confidence: 0.7 }
        : { risk_level: "none", reason: "none", confidence: 0.9 };
      
      // Update risk level in database
      const riskLevel = s.crisis ? "high" : "none";
      await callService.updateCall(callId, { riskLevel });
      
      return res.send({ ok: true, result });
    }

    return res.send({ ok: true, result: {} });
  });

  // -------------------------------
  // Call end
  // -------------------------------
  app.post("/vapi/call-end", async (req, res) => {
    const body: any = req.body || {};
    const callId = body.callId;
    if (!callId) return res.code(400).send({ ok: false, error: "missing_callId" });

    const s = sessions.get(callId);

    // End the call with comprehensive data
    try {
      await callService.endCall(callId, {
        endReason: body.reason || body.end_reason || "normal",
        durationSeconds: Number(body.duration_s || 0) || undefined,
        endedAt: new Date(),
      });
      
      // Update final session data
      if (s) {
        await callService.updateCall(callId, {
          lang: s.lang,
          crisis_flag: s.crisis,
          kb_used: s.kbUses > 0,
          kb_count: s.kbUses,
          turnCount: s.turnCount,
        });
      }
    } catch (error) {
      req.log.error({ error, callId }, "Failed to end call record");
    }

    // Fallback: build chronological transcript even if end-of-call-report didn't arrive
    try { await buildAndStoreChronologicalTranscript(callId); } catch {}

    sessions.delete(callId);
    return res.code(200).send({ ok: true });
  });

  // -------------------------------
  // Store end-of-call report (summary + stitched transcript)
  // -------------------------------
  app.post("/vapi/end-of-call-report", async (req, res) => {
    const raw: any = req.body ?? {};

    const callId =
      raw.callId ??
      raw.call?.id ??
      raw.message?.call?.id ??
      raw.message?.conversation?.callId ??
      raw.message?.callId ??
      raw.id ??
      "unknown";

    if (!callId) return res.code(400).send({ ok: false, error: "missing_callId" });

    const summary =
      raw.message?.analysis?.summary ??
      raw.analysis?.summary ??
      null;

    const stitchedTranscript =
      raw.message?.transcript ??
      raw.transcript ??
      raw.message?.artifact?.transcript ??
      null;

    try {
      const db = await getDb();

      // Always store the summary & timestamps as metadata
      const update: any = {
        $set: {
          endedAt: new Date(),
          endOfCallReportAt: new Date(),
          reportSummary: summary,
        },
      };

      // Only store the big stitched transcript if consented
      const call = await db.collection("calls").findOne(
        { callId },
        { projection: { transcriptOptIn: 1 } }
      );

      if (call?.transcriptOptIn && stitchedTranscript) {
        const clean = redactPII(stitchedTranscript);
        const { ciphertext, iv, tag } = encryptMaybe(clean);

        if (CFG.enc.enabled) {
          update.$set.fullTranscriptEnc = { ciphertext, iv, tag };
        } else {
          update.$set.fullTranscript = clean;
        }
        // NEW: Populate utterances from stitched transcript (authoritative), then build chrono transcript
        await upsertUtterancesFromStitchedTranscript(callId, stitchedTranscript);
      }

      // Optional: keep raw for audit/debug (remove in prod if not needed)
      update.$set.rawEndOfCall = raw;

      await db.collection("calls").updateOne({ callId }, update, { upsert: true });

      // Build + store the chronological (user/assistant) transcript
      await buildAndStoreChronologicalTranscript(callId);

      req.log.info({ callId }, "end-of-call-report stored");
      return res.code(200).send({ ok: true });
    } catch (e) {
      req.log.error({ e, raw }, "failed to store end-of-call-report");
      // ACK to avoid retries; logs will alert us
      return res.code(200).send({ ok: true });
    }
  });

  // -------------------------------
  // Escalation
  // -------------------------------
  app.post("/escalate", async (req, res) => {
    const body: any = req.body || {};
    const callId = body.callId;
    const reason = body.reason || "crisis_detected";
    
    try {
      await callService.escalateCall(callId, reason);
    } catch (error) {
      req.log.error({ error, callId }, "Failed to record escalation");
    }
    
    await vapi.escalate(callId, CFG.hotlineNumber);
    return res.code(200).send({ ok: true });
  });

  // -------------------------------
  // Webhook fan-out
  // -------------------------------
  app.post("/vapi/webhook", async (req, res) => {
    if (!verify(req, res)) return;

    const raw: any = req.body ?? {};
    const msg: any = raw?.message ?? raw;


    const type =
      msg?.type ??
      msg?.event ??
      msg?.event_type ??
      raw?.type ??
      raw?.event ??
      raw?.event_type ??
      "unknown";

    const callId: string =
      msg?.callId ??
      msg?.call_id ??
      msg?.id ??
      msg?.call?.id ??
      msg?.conversation?.callId ??
      raw?.callId ??
      raw?.call_id ??
      raw?.id ??
      "unknown";

    lastEventAt = Date.now();

    req.log.info({ type, callId, keys: Object.keys(raw || {}) }, "Vapi webhook received");
    res.header("X-Orchestrator", "ai-voice");

    // Dispatches
    if (type === "call.started" || type === "on_call_start" || type === "assistant.started") {
      return injectAndAck(app, req, res, { method: "POST", url: "/vapi/call-start", payload: raw.message?.all ? raw.message.call : { callId } });

    }

    if (type === "user.input" || type === "on_user_input") {
      const payload = {
        callId,
        intent: raw.intent,
        transcript: raw.transcript?.text ?? raw.transcript ?? "",
        isFinal: false,
      };
      return injectAndAck(app, req, res, { method: "POST", url: "/vapi/user-input", payload });
    }

    if (type === "transcript.partial") {
      return res.code(200).send({ ok: true });
    }

    // Streaming speech chunks (users + assistant)
    if (type === "speech-update" && raw.message) {
      const m = raw.message;
      const role = m.role || m.speaker || m.source || "user";
      const status = String(m.status || m.stage || m.state || "").toLowerCase();

      const transcript =
        m.transcript?.text ??
        m.transcript ??
        m.artifact?.transcript?.text ??
        m.artifact?.text ??
        m.text ??
        "";

      const isFinal =
        /final|complete|completed|commit|endpoint/.test(status) || m.isFinal === true;

      if (role === "user" && transcript && isFinal) {
        req.log.info({ callId, transcript }, "Final user transcript (speech-update)");
        const payload = { callId, transcript, intent: m.intent, isFinal: true};
        return injectAndAck(app, req, res, { method: "POST", url: "/vapi/user-input", payload });
      }

      if (role === "assistant" && transcript && isFinal) {
        try { await saveAssistantUtterance(callId, transcript, true); } catch (e) {
          req.log.error({ e, callId }, "Failed to store assistant utterance");
        }
      }

      return res.code(200).send({ ok: true });

    }

    // Legacy transcript events (finals)
    if (type === "transcript.final" || type === "transcript.partial") {
      const transcript = raw.transcript?.text ?? raw.transcript ?? "";
      const payload = {callId, transcript, intent: raw.intent, isFinal: true };
      if (type === "transcript.final" && transcript) {
        req.log.info({ callId, transcript }, "Final user transcript (transcript.final)");

        return injectAndAck(app, req, res, { method: "POST", url: "/vapi/user-input", payload });
      }
      return res.code(200).send({ ok: true });

    }
  // Conversation batch updates (can include both roles)


  // conversation-update (captures BOTH user + assistant reliably)
  // IMPORTANT: do this once at the top of /vapi/webhook (recommended)

  if (type === "conversation-update") {
    try {
      //  Payload normalization (local, in-case you haven't done it globally yet)
      const msg: any = raw?.message ?? raw;

      // Extract text robustly from various shapes Vapi may send
      const extractText = (c: any): string => {
        if (!c) return "";
        if (typeof c === "string") return c.trim();
        if (Array.isArray(c)) return c.map(extractText).filter(Boolean).join(" ").trim();
        if (typeof c === "object") {
          return String(
            c.text ??
              c.content ??
              c.transcript?.text ??
              c.transcript ??
              c.message ??
              ""
          ).trim();
        }
        return String(c).trim();
      };

      // use msg not raw.message
      const convo = msg?.conversation ?? msg?.message?.conversation;
      if (!Array.isArray(convo) || convo.length === 0) {
        return res.code(200).send({ ok: true });
      }

      const last = convo[convo.length - 1] as any;

      const role = String(last?.role ?? last?.speaker ?? last?.source ?? "").toLowerCase();
      const status = String(last?.status ?? last?.stage ?? last?.state ?? "").toLowerCase();
      const looksFinal =
        !status || /final|completed|complete|commit|endpoint/.test(status) || last?.isFinal === true;

      // Vapi often puts text in `content`, but sometimes elsewhere
      const text = extractText(last?.content ?? last?.message ?? last?.transcript ?? last);

      if (looksFinal && text) {
        if (role === "user") {
          req.log.info({ callId, text }, "Final user transcript (conversation-update)");

          // NOTE: injectAndAck must accept method typed as Fastify's HTTPMethods.
          return injectAndAck(app, req, res, {
            method: "POST",
            url: "/vapi/user-input",
            payload: { callId, transcript: text, isFinal: true },
          });
        }

        if (role === "assistant") {
          req.log.info({ callId, text }, "Final assistant transcript (conversation-update)");
          try {
            await saveAssistantUtterance(callId, text, true);
          } catch (e) {
            req.log.error({ e, callId }, "Failed to store assistant utterance (conversation-update)");
          }
          return res.code(200).send({ ok: true });
        }
      }
    } catch (e: any) {
      req.log.error({ err: e?.message, callId }, "conversation-update handling failed");
    }

    return res.code(200).send({ ok: true });
  }

    if (type === "tool.call" || type === "on_tool_call") {
      const payload = {
        callId,
        tool: raw.tool?.name || raw.tool,
        query: raw.tool?.args?.query ?? raw.query,
      };
       
      return injectAndAck(app, req, res, { method: "POST", url: "/vapi/tool-call", payload });

    }

    if (type === "call.ended" || type === "on_call_end") {
      const payload = {
        callId,
        ts_start: raw.ts_start,
        duration_s: raw.duration_s,
        reason: raw.reason || raw.end_reason,
      };

      return injectAndAck(app, req, res, { method: "POST", url: "/vapi/call-end", payload });
    }

    if (type === "end-of-call-report") {
      const payload = raw;
      return injectAndAck(app, req, res, { method: "POST" as const, url: "/vapi/end-of-call-report", payload });
    }

    if (type === "status-update") {
      return res.code(200).send({ ok: true });
    }

    req.log.warn({ type, callId, raw }, "Unhandled Vapi webhook event");
    return res.code(200).send({ ok: true });
  });

  // -------------------------------
  // Call management endpoints
  // -------------------------------
  app.get("/calls/metrics", async (req, res) => {
    try {
      const metrics = await callService.getCallMetrics();
      return res.send(metrics);
    } catch (error) {
      req.log.error({ error }, "Failed to get call metrics");
      return res.code(500).send({ error: "Failed to get metrics" });
    }
  });

  app.get("/calls/recent", async (req, res) => {
    try {
      const q = req.query as { limit?: string | number };
      const limit = Number(q?.limit ?? 50) || 50;

      const calls = await callService.getRecentCalls(limit);
      return res.send({ calls });
    } catch (error) {
      req.log.error({ error }, "Failed to get recent calls");
      return res.code(500).send({ error: "Failed to get calls" });
    }
  });

  app.get("/calls/:callId", async (req, res) => {
    try {
      const p = req.params as { callId: string };
      const callId = p.callId;

      const call = await callService.getCall(callId);
      if (!call) {
        return res.code(404).send({ error: "Call not found" });
      }
      return res.send(call);
    } catch (error) {
      req.log.error({ error }, "Failed to get call");
      return res.code(500).send({ error: "Failed to get call" });
    }
  });

  app.post("/calls/:callId/remarks", async (req, res) => {
    try {
      const p = req.params as { callId: string };
      const callId = p.callId;

      const { remarks, tags } = req.body as { remarks: string; tags?: string[] };
      
      await callService.addRemarks(callId, remarks, tags);
      return res.code(200).send({ ok: true });
    } catch (error) {
      req.log.error({ error }, "Failed to add remarks");
      return res.code(500).send({ error: "Failed to add remarks" });
    }
  });

  app.get("/vapi/last-event", async () => ({
    lastEventAt,
    iso: lastEventAt ? new Date(lastEventAt).toISOString() : null,
  }));
};

export default registerVapiRoutes;