"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CallService = void 0;
const db_js_1 = require("../db.js");
const shared_1 = require("@starter/shared");
class CallService {
    static getInstance() {
        if (!CallService.instance) {
            CallService.instance = new CallService();
        }
        return CallService.instance;
    }
    async callsCol() {
        const db = await (0, db_js_1.getDb)();
        return db.collection("calls");
    }
    async createCall(callData) {
        try {
            const db = await (0, db_js_1.getDb)();
            const now = new Date();
            const callRecord = {
                ...callData,
                callStatus: "active",
                crisis_flag: false,
                kb_used: false,
                kb_count: 0,
                kb_opt_in: false,
                turnCount: 0,
                transcriptOptIn: false, // default to false, require explicit opt-in
                createdAt: now,
                updatedAt: now,
                version: 1,
            };
            await db.collection("calls").insertOne(callRecord);
            shared_1.logger.info({ callId: callData.callId }, "Call record created");
        }
        catch (error) {
            shared_1.logger.error({ error, callId: callData.callId }, "Failed to create call record");
            throw error;
        }
    }
    // CallService.updateCall(callId, patch, inc)
    async updateCall(callId, patch = {}, inc = {}) {
        const db = await (0, db_js_1.getDb)();
        const update = { $set: { ...patch, updatedAt: new Date() } };
        if (inc && Object.keys(inc).length)
            update.$inc = inc;
        await db.collection("calls").updateOne({ callId }, update, { upsert: true });
    }
    async endCall(callId, endData) {
        try {
            const updates = {
                callStatus: "completed",
                endedAt: endData.endedAt || new Date(),
                endReason: endData.endReason || "normal",
                durationSeconds: endData.durationSeconds,
            };
            await this.updateCall(callId, updates);
            shared_1.logger.info({ callId, endReason: endData.endReason }, "Call ended");
        }
        catch (error) {
            shared_1.logger.error({ error, callId }, "Failed to end call");
            throw error;
        }
    }
    async escalateCall(callId, reason) {
        try {
            const updates = {
                escalated: true,
                escalatedAt: new Date(),
                escalationReason: reason,
                callStatus: "escalated",
                riskLevel: "high",
            };
            await this.updateCall(callId, updates);
            shared_1.logger.info({ callId, reason }, "Call escalated");
        }
        catch (error) {
            shared_1.logger.error({ error, callId }, "Failed to escalate call");
            throw error;
        }
    }
    async getCall(callId) {
        try {
            const db = await (0, db_js_1.getDb)();
            const calls = await this.callsCol();
            return await calls.findOne({ callId });
            // return await db.collection("calls").findOne({ callId });
        }
        catch (error) {
            shared_1.logger.error({ error, callId }, "Failed to get call record");
            return null;
        }
    }
    async getCallMetrics(timeRange) {
        try {
            const db = await (0, db_js_1.getDb)();
            const collection = db.collection("calls");
            const matchStage = timeRange
                ? { startedAt: { $gte: timeRange.start, $lte: timeRange.end } }
                : {};
            const pipeline = [
                { $match: matchStage },
                {
                    $group: {
                        _id: null,
                        totalCalls: { $sum: 1 },
                        activeCalls: { $sum: { $cond: [{ $eq: ["$callStatus", "active"] }, 1, 0] } },
                        completedCalls: { $sum: { $cond: [{ $eq: ["$callStatus", "completed"] }, 1, 0] } },
                        failedCalls: { $sum: { $cond: [{ $eq: ["$callStatus", "failed"] }, 1, 0] } },
                        escalatedCalls: { $sum: { $cond: ["$escalated", 1, 0] } },
                        avgDuration: { $avg: "$durationSeconds" },
                        crisisCalls: { $sum: { $cond: ["$crisis_flag", 1, 0] } },
                        kbUsedCalls: { $sum: { $cond: ["$kb_used", 1, 0] } },
                        transcriptOptInCalls: { $sum: { $cond: ["$transcriptOptIn", 1, 0] } },
                    }
                }
            ];
            const result = await collection.aggregate(pipeline).toArray();
            const stats = result[0] || {};
            return {
                totalCalls: stats.totalCalls || 0,
                activeCalls: stats.activeCalls || 0,
                completedCalls: stats.completedCalls || 0,
                failedCalls: stats.failedCalls || 0,
                escalatedCalls: stats.escalatedCalls || 0,
                averageDuration: Math.round(stats.avgDuration || 0),
                crisisRate: stats.totalCalls ? (stats.crisisCalls / stats.totalCalls) * 100 : 0,
                kbUsageRate: stats.totalCalls ? (stats.kbUsedCalls / stats.totalCalls) * 100 : 0,
                transcriptOptInRate: stats.totalCalls ? (stats.transcriptOptInCalls / stats.totalCalls) * 100 : 0,
            };
        }
        catch (error) {
            shared_1.logger.error({ error }, "Failed to get call metrics");
            throw error;
        }
    }
    async getRecentCalls(limit = 50) {
        try {
            const calls = await this.callsCol();
            return await calls
                .find({})
                .sort({ startedAt: -1 })
                .limit(limit)
                .toArray();
        }
        catch (error) {
            shared_1.logger.error({ error }, "Failed to get recent calls");
            return [];
        }
    }
    async addRemarks(callId, remarks, tags) {
        try {
            const updates = { remarks };
            if (tags) {
                updates.tags = tags;
            }
            await this.updateCall(callId, updates);
            shared_1.logger.info({ callId }, "Remarks added to call");
        }
        catch (error) {
            shared_1.logger.error({ error, callId }, "Failed to add remarks");
            throw error;
        }
    }
}
exports.CallService = CallService;
